#!/usr/bin/env python3
"""
jobseekusa.nichesite.org
jobseekusa
This script generates a static homepage (index.html) using the provided template.
It reads job metadata from a JSON index file (jobs_index.json) in the job directory,
selects 20 jobs, and renders them into the homepage template using Jinja2.
The homepage includes a client-side search that updates the URL with a "search" query parameter.
 
Usage:
    python generate_homepage.py --jobs_dir /path/to/job --output_dir /path/to/output
"""

import os
import json
import argparse
from jinja2 import Template

def load_jobs(jobs_dir, index_filename="jobs_index.json"):
    """Load job metadata from the JSON index file located in jobs_dir."""
    index_path = os.path.join(jobs_dir, index_filename)
    with open(index_path, "r", encoding="utf-8") as f:
        jobs = json.load(f)
    return jobs

def main(jobs_dir, output_dir):
    # Load all job metadata
    all_jobs = load_jobs(jobs_dir)
    
    # Select 20 jobs (adjust logic if needed)
    selected_jobs = all_jobs[:20]
    
    # Simplified Homepage template
    template_home = r"""<!DOCTYPE html>
<html lang="en">
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>jobseekusa | Remote Jobs</title>
        
        <style>
            body {
                font-family: Arial, sans-serif;
                background-color: #f4f4f9;
                color: #333;
                line-height: 1.6;
                margin: 0;
            }
            .container {
                max-width: 900px;
                margin: 0 auto;
                padding: 0 15px;
            }
            header {
                background-color: #2c3e50; /* Dark blue/gray */
                color: white;
                padding: 20px 0;
                text-align: center;
                margin-bottom: 20px;
            }
            header a {
                color: white;
                text-decoration: none;
            }
            h1 {
                font-size: 2em;
                margin-bottom: 5px;
                font-weight: bold;
            }
            .search-form {
                max-width: 500px;
                margin: 20px auto;
                display: flex;
                gap: 5px;
            }
            .search-input {
                flex: 1;
                padding: 10px;
                border: 1px solid #ccc;
                border-radius: 4px;
                font-size: 1em;
            }
            .search-button {
                padding: 10px 15px;
                background-color: #3498db; /* Blue */
                color: white;
                border: none;
                border-radius: 4px;
                cursor: pointer;
                font-size: 1em;
                transition: background-color 0.2s;
            }
            .search-button:hover {
                background-color: #2980b9;
            }
            .jobs-grid { 
                display: grid;
                gap: 15px;
            }
            @media (min-width: 640px) {
                .jobs-grid {
                    grid-template-columns: repeat(2, 1fr);
                }
            }
            .job-card {
                background: white;
                padding: 15px;
                border: 1px solid #ddd;
                border-radius: 4px;
                display: block;
                transition: border-color 0.2s;
            }
            .job-card:hover {
                border-color: #3498db;
            }
            .job-title {
                font-size: 1.1em;
                font-weight: bold;
                color: #2c3e50;
                margin-bottom: 5px;
            }
            .job-meta {
                font-size: 0.9em;
                color: #777;
            }
            .no-results {
                text-align: center;
                padding: 30px;
                background: white;
                border: 1px solid #ddd;
                border-radius: 4px;
            }
            footer {
                text-align: center;
                padding: 15px 0;
                color: #777;
                border-top: 1px solid #eee;
                margin-top: 30px;
            }
        </style>

        <meta name="description" content="Find your next remote job in the USA.">
        <meta name="keywords" content="remote jobs, work from home, USA jobs">
        <meta property="og:url" content="https://jobseekusa.nichesite.org">
    </head>
    <body>
        <header>
            <div class="container">
                <a href="/">
                    <h1>Remote Jobs USA</h1>
                </a>
                <p>Find your next remote opportunity easily.</p>
            </div>
        </header>

        <main class="container">
            <form id="searchForm" action="/" method="GET" class="search-form">
                <input 
                    type="search" 
                    name="search" 
                    placeholder="Search remote jobs..." 
                    value=""
                    class="search-input"
                >
                <button type="submit" class="search-button">
                    Search
                </button>
            </form>

            <div class="jobs-grid">
                {% for job in jobs %}
                <a href="{{ job.job_url }}" class="job-card">
                    <div class="job-title">{{ job.job_title }}</div>
                    <div class="job-meta">
                        <span>{{ job.location }}</span> &bull; 
                        <span>{{ job.employment_type }}</span>
                    </div>
                </a>
                {% endfor %}
            </div>
            
            </main>

        <footer>
            <div class="container">
                <p>&copy; jobseekusa</p>
            </div>
        </footer>

        <script>
            (function () {
                const jobsIndexUrl = 'job/jobs_index.json';
                let allJobs = [];
                const jobsGrid = document.querySelector('.jobs-grid');
                const searchForm = document.querySelector('#searchForm');
                const searchInput = document.querySelector('.search-input');

                function updateURLParameter(query) {
                    const url = new URL(window.location);
                    if (query) {
                        url.searchParams.set('search', query);
                    } else {
                        url.searchParams.delete('search');
                    }
                    window.history.pushState({}, '', url);
                }

                function renderJobs(jobs) {
                    jobsGrid.innerHTML = '';
                    if (jobs.length === 0) {
                        const noResults = document.createElement('div');
                        noResults.className = 'no-results';
                        noResults.innerHTML = '<p>No jobs found matching your search.</p>';
                        jobsGrid.appendChild(noResults);
                        return;
                    }
                    jobs.forEach(job => {
                        const card = document.createElement('a');
                        card.href = job.job_url;
                        card.className = 'job-card';
                        card.innerHTML = `
                            <div class="job-title">${job.job_title}</div>
                            <div class="job-meta">
                                <span>${job.location}</span> &bull; 
                                <span>${job.employment_type}</span>
                            </div>
                        `;
                        jobsGrid.appendChild(card);
                    });
                }

                function loadJobs() {
                    fetch(jobsIndexUrl)
                        .then(response => {
                            if (!response.ok) {
                                throw new Error('Network response was not ok: ' + response.status);
                            }
                            return response.json();
                        })
                        .then(data => {
                            allJobs = data;
                            const params = new URLSearchParams(window.location.search);
                            const initialQuery = params.get('search') || "";
                            searchInput.value = initialQuery;
                            const filteredJobs = performSearch(initialQuery);
                            renderJobs(filteredJobs);
                        })
                        .catch(error => {
                            console.error('Error loading jobs index:', error);
                        });
                }

                function performSearch(query) {
                    const q = (query || searchInput.value).toLowerCase().trim();
                    return allJobs.filter(job =>
                        job.job_title.toLowerCase().includes(q) ||
                        job.job_url.toLowerCase().includes(q)
                    ).slice(0, 20);
                }

                searchForm.addEventListener('submit', function (event) {
                    event.preventDefault();
                    const query = searchInput.value;
                    updateURLParameter(query);
                    renderJobs(performSearch(query));
                });

                searchInput.addEventListener('input', function () {
                    const query = searchInput.value;
                    updateURLParameter(query);
                    renderJobs(performSearch(query));
                });

                loadJobs();
            })();
        </script>
    </body>
</html>
"""

    # Render the homepage template with the selected 20 jobs.
    template = Template(template_home)
    rendered_html = template.render(jobs=selected_jobs)

    # Create the output directory if it does not exist.
    os.makedirs(output_dir, exist_ok=True)
    output_path = os.path.join(output_dir, "index.html")
    with open(output_path, "w", encoding="utf-8") as f:
        f.write(rendered_html)
    
    print(f"Homepage generated at: {output_path}")

if __name__ == "__main__":
    parser = argparse.ArgumentParser(
        description="Generate homepage (index.html) using 20 jobs from the job directory and include search functionality with URL parameters."
    )
    parser.add_argument("--jobs_dir", required=True, help="Path to the job directory (should contain jobs_index.json)")
    parser.add_argument("--output_dir", required=True, help="Output directory for the generated index.html")
    args = parser.parse_args()
    main(args.jobs_dir, args.output_dir)