import os
import json
from bs4 import BeautifulSoup

def extract_job_info(file_path):
    """
    Extracts job information from a single HTML file.
    Assumes the job title is in the <title> element.
    Generates the job URL from the file name.
    """
    with open(file_path, "r", encoding="utf-8") as f:
        html = f.read()
    
    soup = BeautifulSoup(html, "html.parser")
    # Get the job title from the <title> tag if available.
    title_tag = soup.find("title")
    job_title = title_tag.text.strip() if title_tag else "No Title Found"
    
    # Generate the job URL from the file name (remove .html extension and prepend "/job/")
    file_name = os.path.basename(file_path)
    slug = file_name[:-5] if file_name.lower().endswith(".html") else file_name
    job_url = f"/job/{slug}"
    
    # Set location - here all jobs are assumed to be remote.
    location = "Remote, USA"
    
    # Determine employment type based on the job title.
    lower_title = job_title.lower()
    if "part-time" in lower_title or "part time" in lower_title:
        employment_type = "Part-time"
    else:
        employment_type = "Full-time"
        
    return {
        "job_title": job_title,
        "job_url": job_url,
        "location": location,
        "employment_type": employment_type
    }

def main():
    # List all HTML files in the current directory.
    html_files = [f for f in os.listdir(".") if f.lower().endswith(".html")]
    
    jobs = []
    for file in html_files:
        job_data = extract_job_info(file)
        jobs.append(job_data)
    
    # Write the JSON output to output.json
    with open("jobs_index.json", "w", encoding="utf-8") as out_file:
        json.dump(jobs, out_file, indent=2)
    
    print("Output written to output.json")

if __name__ == "__main__":
    main()
